/*
	Series_To_Parallel_Conversion_Calculator
 	Copyright © 2020 Harry Whitfield

 	This program is free software; you can redistribute it and/or modify it
 	under the terms of the GNU General Public License as published by the
 	Free Software Foundation; either version 2 of the License, or (at your
 	option) any later version.

 	This program is distributed in the hope that it will be useful, but
 	WITHOUT ANY WARRANTY; without even the implied warranty of
 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 	General Public License for more details.

 	You should have received a copy of the GNU General Public License along
 	with this program; if not, write to the Free Software Foundation, Inc.,
 	51 Franklin St, Fifth Floor, Boston, MA	 02110-1301	 USA

 	Series_To_Parallel_Conversion_Calculator - browser version 1.0x
 	31 May, 2020
 	Copyright © 2020 Harry Whitfield
 	mailto:g6auc@arrl.net

	This version has been modified to facilitate future enhancements.
*/

/*jslint browser, devel, for, this */

/*property
    G, M, PI, T, background, focus, isNaN, k, keyCode, length, m, match, n,
    onkeydown, onkeypress, p, pow, preventDefault, replace, select, style,
    title, toFixed, toString, u, value, which
*/

import {paramOne, paramTwo, paramThree, paramFour} from "./model.js";
import {headingOne, headingTwo, headingThree, headingFour} from "./model.js";
import {title} from "./model.js";

//var gWidgetName;	// global

// map custom names onto model names
var sR = paramOne;
var sX = paramTwo;
var pR = paramThree;
var pX = paramFour;

var sRHeading = headingOne;
var sXHeading = headingTwo;
var pRHeading = headingThree;
var pXHeading = headingFour;

//var title;
title.value = "Series to Parallel Conversion";

var suffixes = "\nYou may append one of the SI suffixes";
suffixes += "\nT, G, M, k, m, µ/u, n or p";
suffixes += "\nfollowed by an optional Ω character.";

sR.title = "Enter the series resistance." + suffixes;
sX.title = "Enter the series reactance." + suffixes;
pR.title = "Enter the parallel resistance." + suffixes;
pX.title = "Enter the parallel reactance." + suffixes;

sRHeading.value = "Series Resistance";
sXHeading.value = "Series Reactance";
pRHeading.value = "Parallel Resistance";
pXHeading.value = "Parallel Reactance";

var s2p = true;
var prec = 3;

function cToX(f, c) {
	return 1 / (2 * Math.PI * f * c);
}

function lToX(f, l) {
	return 2 * Math.PI * f * l;
}

function xToC(f, x) {
	return 1 / (2 * Math.PI * f * x);
}

function xToL(f, x) {
	return x / (2 * Math.PI * f);
}

function decodeSI(s) {	// now allows for a unit suffix F|Hz|H|Ω
	const table = {"T": 1e12, "G": 1e9, "M": 1e6, "k": 1e3, "Ω": 1, "m": 1e-3, "µ": 1e-6, "u": 1e-6, "n": 1e-9, "p": 1e-12};
	const ok = /^-?\d+(\.\d+)?([eE]-?\d{1,2})?[TGMkmµunp]?(F|Hz|H|Ω)?$/;
	var found;
	var unit;	// unit suffix
	var uLen = 0;
	var val;
	var suffix;	// multiplier suffix
	var power;

	s = s.replace(/\s/g, "");

	if (s === "Infinity") {
		return [1e99];
	}
	if (s === "-Infinity") {
		return [-1e99];
	}

	found = s.match(ok);
	if (found === null) {
		return [NaN];
	}

	unit = found[3];
	if (unit !== undefined) {
		uLen = unit.length;
	}

	val = parseFloat(s);
	suffix = s[s.length - 1 - uLen];
	power = table[suffix];

	if (power !== undefined) {
		val *= power;
	}
	return [val, unit];
}

function encodeSI(n, unit) {
	const table = ["T", "G", "M", "k", "", "m", "µ", "n", "p"];
	var i;
	var j;
	var p;
	var tmp;
	var sign = "";
	var res;

	unit = unit || "Ω";

	if (n === Infinity) {
		return Infinity;
	}

	if (n < 0) {
		sign = "-";
		n = -n;
	}

	for (i = 4; i >= -4; i -= 1) {
		j = 3 * i;
		p = Math.pow(10, j);
		if (n >= p) {
			tmp = n / p;
			res = tmp.toFixed(prec) + table[4 - i] + unit;
			return (
				sign === "-"
				? sign + res
				: res
			);
		}
	}
	return (
		sign === "-"
		? -n.toString() + unit
		: n.toString() + unit
	);
}

function process() {
    var rS;
    var rU;
    var xS;
    var xU;
    var rP;
    var xP;
    var ms;
    var res;

	if (s2p) {
		res = decodeSI(sR.value);
		rS = res[0];
		rU = res[1];
		res = decodeSI(sX.value);
		xS = res[0];
		xU = res[1];

		if (rU && (rU !== "Ω")) {
        	pR.value = "Invalid Series Resistance";
        	return;
        }
   		if (Number.isNaN(rS)) {
        	pR.value = "Invalid Series Resistance";
        	return;
    	}
    	if (rS < 0) {
    		pR.value = "Invalid Series Resistance";
    		return;
    	}

//		if (xU && (xU !== "Ω") && (xU !== "H") && (xU !== "F")) {
		if (xU && (xU !== "Ω")) {
        	pX.value = "Invalid Series Reactance";
        	return;
        }
    	if (Number.isNaN(xS)) {
        	pX.value = "Invalid Series Reactance";
        	return;
    	}

    	ms = rS * rS + xS * xS;
     	pR.value = encodeSI(ms / rS);
    	pX.value = encodeSI(ms / xS);
    } else {
		res = decodeSI(pR.value);
		rP = res[0];
		rU = res[1];
		res = decodeSI(pX.value);
		xP = res[0];
		xU = res[1];

		if (rU && (rU !== "Ω")) {
        	sR.value = "Invalid Parallel Resistance";
        	return;
        }
    	if (Number.isNaN(rP)) {
        	sR.value = "Invalid Parallel Resistance";
        	return;
    	}
    	if (rP < 0) {
    		sR.value = "Invalid Parallel Resistance";
    		return;
    	}

//		if (xU && (xU !== "Ω") && (xU !== "H") && (xU !== "F")) {
		if (xU && (xU !== "Ω")) {
        	pX.value = "Invalid Parallel Reactance";
        	return;
        }
    	if (Number.isNaN(xP)) {
        	sX.value = "Invalid Parallel Reactance";
        	return;
    	}

		ms = rP * rP + xP * xP;
		sR.value = encodeSI(rP * xP * xP / ms);
		sX.value = encodeSI(xP * rP * rP / ms);
    }
}

function setBackgrounds() {
	const lite = "rgba(255 ,255, 255, 1.0)";
	const dark = "rgba(192 ,192, 192, 1.0)";

	if (s2p) {
		sR.style.background = lite;
		sX.style.background = lite;
		pR.style.background = dark;
		pX.style.background = dark;
	} else {
		sR.style.background = dark;
		sX.style.background = dark;
		pR.style.background = lite;
		pX.style.background = lite;
	}
}

function test_case() {
	s2p = true;
	setBackgrounds();
	sR.value = "10Ω";
	sX.value = "-20Ω";
	process();
}

var Return = 13;
var Tab = 9;
//var Backspace = 8;

sR.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	s2p = true;
	setBackgrounds();

	if (x === Return) {
		process();
    } else {
        pR.value = "";
        pX.value = "";
    }
};

sX.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	s2p = true;
	setBackgrounds();

	if (x === Return) {
		process();
    } else {
        pR.value = "";
        pX.value = "";
    }
};

pR.onkeypress = function (event) {
	var x = event.which || event.keyCode;

	s2p = false;
	setBackgrounds();

	if (x === Return) {
		process();
    } else {
        sR.value = "";
        sX.value = "";
    }
};

pX.onkeydown = function (event) {
	var x = event.which || event.keyCode;

	s2p = false;
	setBackgrounds();

	if (x === Tab) {
		event.preventDefault();
		sR.focus();
		sR.select();
	} else if (x === Return) {
		process();
    } else {
        sR.value = "";
        sX.value = "";
    }
};

//gWidgetName = "Series_To_Parallel_Conversion Web Widget\n\n";
test_case();
